package nemosofts.notes.app.utils.helper;

import android.content.Context;
import android.net.Uri;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import org.apache.commons.io.FileUtils;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * Simple backup and restore helper that zips the database and media folders into a single file.
 */
public class BackupManager {

    private static final String TAG = "BackupManager";
    private static final int BUFFER_SIZE = 16 * 1024;
    private static final String ENTRY_DB_PREFIX = "database/";
    private static final String ENTRY_FILES_PREFIX = "files/";

    private BackupManager() {
        // no-op
    }

    public static boolean backup(@NonNull Context context, @NonNull Uri destination) {
        try (OutputStream rawOut = context.getContentResolver().openOutputStream(destination);
             BufferedOutputStream bos = new BufferedOutputStream(rawOut);
             ZipOutputStream zos = new ZipOutputStream(bos)) {

            // database
            File dbFile = DBHelper.getDatabaseFile(context);
            addFileToZip(zos, dbFile, ENTRY_DB_PREFIX + dbFile.getName());

            // media
            File imagesDir = new File(context.getFilesDir(), "notes_images");
            addDirectoryToZip(zos, imagesDir, ENTRY_FILES_PREFIX + "notes_images/");

            File audioDir = new File(context.getFilesDir(), "notes_audio");
            addDirectoryToZip(zos, audioDir, ENTRY_FILES_PREFIX + "notes_audio/");

            zos.finish();
            return true;
        } catch (Exception e) {
            Log.e(TAG, "Backup failed", e);
            return false;
        }
    }

    public static boolean restore(@NonNull Context context, @NonNull Uri source) {
        try (InputStream rawIn = context.getContentResolver().openInputStream(source);
             BufferedInputStream bis = new BufferedInputStream(rawIn);
             ZipInputStream zis = new ZipInputStream(bis)) {

            // clear media folders before restoring
            clearDirectory(new File(context.getFilesDir(), "notes_images"));
            clearDirectory(new File(context.getFilesDir(), "notes_audio"));

            ZipEntry entry;
            while ((entry = zis.getNextEntry()) != null) {
                File target = resolveTargetFile(context, entry.getName());
                if (target == null) {
                    zis.closeEntry();
                    continue;
                }
                if (entry.isDirectory()) {
                    // ensure directory exists
                    if (!target.exists() && !target.mkdirs()) {
                        Log.w(TAG, "Unable to create directory " + target.getAbsolutePath());
                    }
                    zis.closeEntry();
                    continue;
                }
                File parent = target.getParentFile();
                if (parent != null && !parent.exists() && !parent.mkdirs()) {
                    Log.w(TAG, "Unable to create parent directory " + parent.getAbsolutePath());
                }
                try (FileOutputStream fos = new FileOutputStream(target)) {
                    copy(zis, fos);
                }
                zis.closeEntry();
            }
            return true;
        } catch (Exception e) {
            Log.e(TAG, "Restore failed", e);
            return false;
        }
    }

    private static void addDirectoryToZip(@NonNull ZipOutputStream zos, @Nullable File dir, @NonNull String prefix) {
        if (dir == null || !dir.exists() || !dir.isDirectory()) {
            return;
        }
        File[] children = dir.listFiles();
        if (children == null) {
            return;
        }
        for (File child : children) {
            String entryName = prefix + child.getName();
            if (child.isDirectory()) {
                addDirectoryToZip(zos, child, entryName + "/");
            } else {
                addFileToZip(zos, child, entryName);
            }
        }
    }

    private static void addFileToZip(@NonNull ZipOutputStream zos, @Nullable File file, @NonNull String entryName) {
        if (file == null || !file.exists()) {
            return;
        }
        try (FileInputStream fis = new FileInputStream(file);
             BufferedInputStream bis = new BufferedInputStream(fis)) {
            zos.putNextEntry(new ZipEntry(entryName));
            copy(bis, zos);
            zos.closeEntry();
        } catch (Exception e) {
            Log.w(TAG, "Skipping file in backup: " + entryName, e);
        }
    }

    @Nullable
    private static File resolveTargetFile(@NonNull Context context, @NonNull String entryName) {
        if (entryName.startsWith(ENTRY_DB_PREFIX)) {
            return DBHelper.getDatabaseFile(context);
        }
        if (entryName.startsWith(ENTRY_FILES_PREFIX)) {
            String relative = entryName.substring(ENTRY_FILES_PREFIX.length());
            return new File(context.getFilesDir(), relative);
        }
        return null;
    }

    private static void clearDirectory(@NonNull File dir) {
        try {
            FileUtils.deleteDirectory(dir);
        } catch (Exception ignored) {
        }
    }

    private static void copy(@NonNull InputStream in, @NonNull OutputStream out) throws Exception {
        byte[] buffer = new byte[BUFFER_SIZE];
        int len;
        while ((len = in.read(buffer)) != -1) {
            out.write(buffer, 0, len);
        }
        out.flush();
    }
}

